#include <Arduino.h>
#include <Adafruit_NeoPixel.h>
#include <ctype.h>

// ===================== Pins =====================
static const uint8_t VALVE_PINS[5] = {2, 3, 4, 5, 6};   // Ventile 1..5
static const uint8_t STRIP_PIN     = 7;                // LED-Strip Daten
static const uint8_t FLASH_PIN     = 12;               // Blitz Trigger

// Nikon Z7II: Zwei-Stufen-Auslöser
static const uint8_t CAM_HALF_PIN  = 13;               // Fokus (Nikon Rot)
static const uint8_t CAM_FULL_PIN  = 10;               // Shutter (Nikon Weiss)

// ===================== LED Strip =====================
static const uint16_t NUM_PIXELS = 11;
Adafruit_NeoPixel strip(NUM_PIXELS, STRIP_PIN, NEO_GRB + NEO_KHZ800);

volatile uint8_t tankMode = 0;
uint16_t chaseIndex = 0;
uint32_t lastChaseMs = 0;
static const uint16_t CHASE_INTERVAL_MS = 80;

// ===================== Sequenz-Daten =====================
struct RowCfg { uint8_t valve; uint16_t start_ms; uint16_t open_ms; };
static const uint8_t MAX_ROWS = 80;
RowCfg rows[MAX_ROWS];
uint8_t rowCount = 0;

uint16_t camDelayMs = 0;
uint16_t flashDelayMs[3] = {0,0,0};
bool flashEnabled[3] = {false,false,false};

uint16_t repeatCount = 1;
uint32_t pauseBetweenRepeatsMs = 0;
volatile bool abortRun = false;

bool camMasterEnabled = true;
bool flashMasterEnabled = true;

// ===================== Serial Line Buffer =====================
static char lineBuf[96];
static uint8_t lineLen = 0;

// Forward decl
static void handleLine(char* raw);
static void pollSerial();
static void abortableDelayMs(uint32_t ms);

// ===================== Hilfsfunktionen =====================
static inline void setAllValves(bool on) {
  for (uint8_t i=0; i<5; i++) digitalWrite(VALVE_PINS[i], on ? HIGH : LOW);
}

static void stripAllOff() {
  for (uint16_t i=0; i<NUM_PIXELS; i++) strip.setPixelColor(i, 0);
  strip.show();
}

static void stripSolid(uint32_t color) {
  for (uint16_t i=0; i<NUM_PIXELS; i++) strip.setPixelColor(i, color);
  strip.show();
}

static void stripChaseTick() {
  uint32_t now = millis();
  if (now - lastChaseMs < CHASE_INTERVAL_MS) return;
  lastChaseMs = now;

  for (uint16_t i=0; i<NUM_PIXELS; i++) strip.setPixelColor(i, 0);
  strip.setPixelColor(chaseIndex % NUM_PIXELS, strip.Color(180,180,180));
  strip.show();
  chaseIndex = (chaseIndex + 1) % NUM_PIXELS;
}

static void applyTankModeImmediate() {
  switch (tankMode) {
    case 0: stripAllOff(); break;
    case 1: stripSolid(strip.Color(180,180,180)); break;
    case 2: stripSolid(strip.Color(0,0,200)); break;
    case 3: stripSolid(strip.Color(200,0,0)); break;
    case 4: /* Chase läuft in loop() */ break;
    default: stripAllOff(); break;
  }
}

// ===================== Scheduler =====================
struct Action { uint32_t t_us; uint8_t pin; uint8_t level; };
static const uint8_t MAX_ACTIONS = 150;
Action actions[MAX_ACTIONS];
uint8_t actionCount = 0;

static void addActionUs(uint32_t t_us, uint8_t pin, uint8_t level) {
  if (actionCount < MAX_ACTIONS) actions[actionCount++] = {t_us, pin, level};
}

static void sortActions() {
  for (uint8_t i=1; i<actionCount; i++) {
    Action key = actions[i];
    int j = (int)i - 1;
    while (j >= 0 && actions[j].t_us > key.t_us) {
      actions[j+1] = actions[j];
      j--;
    }
    actions[j+1] = key;
  }
}

static void buildActionsForOneRun() {
  actionCount = 0;

  // Ventile
  for (uint8_t i=0; i<rowCount; i++) {
    uint8_t v = rows[i].valve;
    if (v < 1 || v > 5) continue;
    uint8_t pin = VALVE_PINS[v-1];
    addActionUs((uint32_t)rows[i].start_ms * 1000UL, pin, HIGH);
    addActionUs((uint32_t)(rows[i].start_ms + rows[i].open_ms) * 1000UL, pin, LOW);
  }

  // Kamera: kein extra Offset, ACTIVE-HIGH wie im Original-Kommentar (Idle LOW)
  if (camMasterEnabled) {
    uint32_t tCamStart = (uint32_t)camDelayMs * 1000UL;
    addActionUs(tCamStart, CAM_HALF_PIN, HIGH);
    addActionUs(tCamStart, CAM_FULL_PIN, HIGH);
    addActionUs(tCamStart + 60000UL, CAM_FULL_PIN, LOW);  // 60ms Puls
    addActionUs(tCamStart + 60000UL, CAM_HALF_PIN, LOW);
  }

  // Blitz
  if (flashMasterEnabled) {
    for (uint8_t i=0; i<3; i++) {
      if (flashEnabled[i]) {
        uint32_t tF = (uint32_t)flashDelayMs[i] * 1000UL;
        addActionUs(tF, FLASH_PIN, HIGH);
        addActionUs(tF + 2000UL, FLASH_PIN, LOW); // 2ms
      }
    }
  }

  sortActions();
}

static void runOneSequenceAccurate() {
  buildActionsForOneRun();

  // Idle-Zustände
  setAllValves(false);
  digitalWrite(FLASH_PIN, LOW);
  digitalWrite(CAM_HALF_PIN, LOW);
  digitalWrite(CAM_FULL_PIN, LOW);

  uint32_t t0 = micros();
  uint8_t idx = 0;

  while (idx < actionCount && !abortRun) {
    uint32_t target = t0 + actions[idx].t_us;

    // Warten bis Zeitpunkt + währenddessen Serial pollen -> NOT AUS sofort
    while ((int32_t)(micros() - target) < 0) {
      pollSerial();
      if (abortRun) break;
    }
    if (abortRun) break;

    digitalWrite(actions[idx].pin, actions[idx].level);
    idx++;
  }

  // Alles aus
  setAllValves(false);
  digitalWrite(FLASH_PIN, LOW);
  digitalWrite(CAM_HALF_PIN, LOW);
  digitalWrite(CAM_FULL_PIN, LOW);
}

// ===================== Parser =====================
static void handleLine(char* raw) {
  // trim leading spaces
  while (*raw == ' ' || *raw == '\t') raw++;

  if (raw[0]=='E' && isdigit((unsigned char)raw[1])) {
    tankMode = (uint8_t)atoi(&raw[1]);
    applyTankModeImmediate();
    Serial.println(F("OK E"));
  }
  else if (!strncmp(raw, "HOLD ", 5)) {
    int v, st;
    if (sscanf(raw+5, "%d %d", &v, &st) == 2 && v>=1 && v<=5) {
      digitalWrite(VALVE_PINS[v-1], st ? HIGH : LOW);
      Serial.println(F("OK HOLD"));
    }
  }
  else if (!strcmp(raw, "CLEAR")) {
    rowCount = 0;
    Serial.println(F("OK CLEAR"));
  }
  else if (!strncmp(raw, "ADD ", 4)) {
    int v, st, op;
    if (sscanf(raw+4, "%d %d %d", &v, &st, &op) == 3) {
      if (rowCount < MAX_ROWS) {
        rows[rowCount++] = {(uint8_t)v, (uint16_t)st, (uint16_t)op};
        Serial.println(F("OK ADD"));
      }
    }
  }
  else if (!strncmp(raw, "REPEAT ", 7)) {
    long n, p;
    int got = sscanf(raw+7, "%ld %ld", &n, &p);
    if (got >= 1) {
      repeatCount = (uint16_t)n;
      pauseBetweenRepeatsMs = (got>=2 && p>0) ? (uint32_t)p : 0;
      Serial.println(F("OK REPEAT"));
    }
  }
  else if (!strcmp(raw, "START") || !strcmp(raw, "RUN")) {
    abortRun = false;

    for (uint16_t r=0; r < repeatCount && !abortRun; r++) {
      runOneSequenceAccurate();
      if (r+1 < repeatCount && pauseBetweenRepeatsMs > 0) {
        abortableDelayMs(pauseBetweenRepeatsMs);
      }
    }
    Serial.println(F("OK START_DONE"));
  }
  else if (!strcmp(raw, "STOP_ALL")) {
    abortRun = true;
    setAllValves(false);
    digitalWrite(FLASH_PIN, LOW);
    digitalWrite(CAM_HALF_PIN, LOW);
    digitalWrite(CAM_FULL_PIN, LOW);
    Serial.println(F("OK STOP_ALL"));
  }
  else if (!strncmp(raw, "CAM_DELAY ", 10)) {
    camDelayMs = (uint16_t)atoi(raw+10);
    Serial.println(F("OK CAM_DELAY"));
  }
  else if (!strncmp(raw, "FLASH_DELAY ", 12)) {
    int ch; long ms;
    if (sscanf(raw+12, "%d %ld", &ch, &ms) == 2 && ch>=1 && ch<=3) {
      flashDelayMs[ch-1] = (uint16_t)ms;
      Serial.println(F("OK FLASH_DELAY"));
    }
  }
  else if (!strncmp(raw, "FLASH_EN ", 9)) {
    int ch, en;
    if (sscanf(raw+9, "%d %d", &ch, &en) == 2 && ch>=1 && ch<=3) {
      flashEnabled[ch-1] = (en != 0);
      Serial.println(F("OK FLASH_EN"));
    }
  }
  else if (!strncmp(raw, "CAM_EN ", 7)) {
    int en = 1;
    sscanf(raw+7, "%d", &en);
    camMasterEnabled = (en != 0);
    Serial.println(F("OK CAM_EN"));
  }
  else if (!strncmp(raw, "FLASH_MASTER_EN ", 16)) {
    int en = 1;
    sscanf(raw+16, "%d", &en);
    flashMasterEnabled = (en != 0);
    Serial.println(F("OK FLASH_MASTER_EN"));
  }
  else {
    // Unknown command – keep silent or respond
    // Serial.println(F("ERR"));
  }
}

// ===================== Serial Polling =====================
static void pollSerial() {
  while (Serial.available() > 0) {
    char c = (char)Serial.read();

    if (c == '\n' || c == '\r') {
      if (lineLen > 0) {
        lineBuf[lineLen] = '\0';
        handleLine(lineBuf);
        lineLen = 0;
      }
    } else {
      if (lineLen < (sizeof(lineBuf) - 1)) {
        lineBuf[lineLen++] = c;
      } else {
        // overflow -> reset line
        lineLen = 0;
      }
    }
  }
}

// ===================== Abortable Delay =====================
static void abortableDelayMs(uint32_t ms) {
  uint32_t t0 = millis();
  while (!abortRun && (uint32_t)(millis() - t0) < ms) {
    pollSerial();
    if (tankMode == 4) stripChaseTick();
    // tiny yield
    delay(1);
  }
}

// ===================== Arduino setup/loop =====================
void setup() {
  Serial.begin(115200);

  for (uint8_t i=0; i<5; i++) {
    pinMode(VALVE_PINS[i], OUTPUT);
    digitalWrite(VALVE_PINS[i], LOW);
  }

  pinMode(FLASH_PIN, OUTPUT);
  digitalWrite(FLASH_PIN, LOW);

  pinMode(CAM_HALF_PIN, OUTPUT);
  pinMode(CAM_FULL_PIN, OUTPUT);
  digitalWrite(CAM_HALF_PIN, LOW);
  digitalWrite(CAM_FULL_PIN, LOW);

  strip.begin();
  strip.show();
  applyTankModeImmediate();

  Serial.println(F("READY"));
}

void loop() {
  pollSerial();

  // Chase Mode tick
  if (tankMode == 4) stripChaseTick();
}
