// =======================================================
//  Z-Achse Tropfenmaschine – Arduino Nano
//
//  Datum: 2026-01-23
//
//  Profi-Finish: Empfehlung für “dauerhaft stabil”
//  1) ENDSTOP_DEBOUNCE_MS fein einstellen
//     Du hast aktuell 15 ms. Das ist ein guter Start.
//     10 ms  → schneller reagieren, etwas empfindlicher
//     15–25 ms → sehr robust gegen Störungen/Vibration
//     >30 ms → kann bei sehr schneller Fahrt schon „spät“ reagieren (meist trotzdem ok)
//
//  Protokoll passend zur GUI
//  - STATUS;STATE_...;moving;fault;end1;end2;pos
//  - U/u/D/d/S/Q  (Jog & Stop & Quittieren)
//  - V<cm/s>      (Geschwindigkeit in cm pro Sekunde)
//  - G<steps>     (absolute Position in Steps)
//  - H            (Homing / Referenzfahrt nach oben)
//  - L            (Servicefahrt nach unten auf unteren Endschalter)
//
//  Homing:
//    - fährt nach oben auf ENDSTOP_2_PIN
//    - setzt currentPositionSteps = 0
//    - fährt ein Stück nach unten (BACKOFF_STEPS) mit HOMING_SPEED_CM_S
//    - danach: homed = true
// =======================================================

// -------------------- OPTIONAL: DEBUG --------------------
// 0 = aus (Normalbetrieb), 1 = an (zusätzliche Debugzeilen in Serial)
#define DEBUG_ENDSTOPS 0

// Debug-Intervall (ms) – nicht zu klein wählen, sonst bremst Serial ggf.
const unsigned long DEBUG_INTERVAL_MS = 200;

// -------------------- PIN-ZUORDNUNG --------------------
#define STEP_PIN        2
#define DIR_PIN         5

#define JOG_UP_PIN      6   // Wippschalter hoch
#define JOG_DOWN_PIN    7   // Wippschalter runter

#define ENDSTOP_1_PIN   4   // unten
#define ENDSTOP_2_PIN   9   // oben

#define LED_GREEN       3   // steht / OK
#define LED_YELLOW      10  // fährt (blinkt langsam)
#define LED_RED         11  // Fehler (blinkt schnell)

#define LED_STATUS      13  // weiße Front-LED (Arduino-Status)

// -------------------- MECHANIK / GESCHWINDIGKEIT --------------------
// Motor: NEMA17, 1.8° -> 200 Vollschritte/Umdrehung
// Treiber: TMC2209, 16 Microsteps
// Spindel: TR10x3 -> 3 mm Steigung pro Umdrehung

const int   FULL_STEPS_PER_REV = 200;
const int   MICROSTEPS         = 16;
const float LEAD_MM            = 3.0f;   // 3 mm/Umdrehung

// daraus:
const float STEPS_PER_MM = (FULL_STEPS_PER_REV * MICROSTEPS) / LEAD_MM;   // ~1066.67
const float STEPS_PER_CM = STEPS_PER_MM * 10.0f;                           // ~10666.67

// Geschwindigkeit kommt aus der GUI in cm/s:
const float V_MIN_CM_S = 0.05f;   // untere Grenze
const float V_MAX_CM_S = 1.10f;   // OBERGRENZE – passend zur GUI

// Homing-Geschwindigkeit (fest)
const float HOMING_SPEED_CM_S = 0.8f;

// aktuelle Fahrgeschwindigkeit (für Jog/Goto), in cm/s
float current_speed_cm_s = 1.0f;   // Startwert (cm/s)

// Schrittzeiten
unsigned long STEP_DELAY_MICROS        = 1000;  // für normale Fahrten
unsigned long homingStepDelayMicros    = 1000;  // für Homing & Homing-Backoff

// Für Rampe (Sanftanlauf)
bool          rampActive               = false;
unsigned long currentStepDelayMicros   = 1000;
const int     RAMP_SMOOTH              = 10;    // je größer, desto weicher
const float   RAMP_START_FACTOR        = 4.0f;  // Start mit 1/4 der Zielgeschwindigkeit

// Backoff: wie weit zurückfahren, wenn Endschalter ausgelöst hat
const int   BACKOFF_REV     = 1;  // 1 Umdrehung
const long  BACKOFF_STEPS   = (long)BACKOFF_REV * FULL_STEPS_PER_REV * MICROSTEPS;
// = 1 * 200 * 16 = 3200 Steps (~4,8 mm)

// -------------------- ENDSTOP-ENTPRELLUNG --------------------
// Endschalter gilt erst als "aktiv", wenn er ENDSTOP_DEBOUNCE_MS stabil LOW ist.
const unsigned long ENDSTOP_DEBOUNCE_MS = 15;

// -------------------- STATUS & ZUSTÄNDE --------------------

enum State {
  STATE_NORMAL = 0,
  STATE_BACKOFF,
  STATE_FAULT_LATCHED,
  STATE_HOMING,
  STATE_SERVICE_HOMING   // Servicefahrt nach unten (unterer Endschalter)
};

State state = STATE_NORMAL;

// Merkt sich, in welche Richtung zuletzt gefahren wurde
bool lastDirectionUp = true;     // true = nach oben, false = nach unten

// Für Backoff
long backoffRemaining   = 0;
bool backoffDirectionUp = false;
bool backoffIsFault     = true;   // true = Fehler-Backoff, false = Homing-Backoff

bool backoffSetHomed  = false;  // nur bei Homing-Backoff: setzt am Ende homed=true
// Für Fehler-Quittierung (Hardware-Wippschalter)
unsigned long faultAckStartTime = 0;
const unsigned long FAULT_ACK_TIME_MS = 5000;

// Für LED-Blinken
const unsigned long YELLOW_PERIOD_MS = 500;  // ~1 Hz
const unsigned long RED_PERIOD_MS    = 125;  // ~4 Hz

unsigned long lastYellowToggle = 0;
bool yellowState = false;

unsigned long lastRedToggle = 0;
bool redState = false;

// Positionszähler (Software-Position, in Steps)
long currentPositionSteps = 0;

// Wurde seit Einschalten erfolgreich gehomed?
bool homed = false;

// -------------------- SERIELLE STEUERUNG --------------------

// Remote-Jog (aus GUI)
bool remoteJogUp   = false;
bool remoteJogDown = false;

// Goto-Fahrt
bool gotoActive       = false;
long gotoTargetSteps  = 0;

// Homing
bool homingActive      = false;
long homingStepsDone   = 0;
const long HOMING_MAX_STEPS = (long)(STEPS_PER_CM * 50.0f);  // Sicherheitsbegrenzung (~50 cm)

// Status-Versand
unsigned long lastStatusMs = 0;
const unsigned long STATUS_INTERVAL_MS = 100;

// -------------------- HILFSFUNKTIONEN --------------------

// Hilfsfunktion: Speed -> Schritt-Delay (µs)
unsigned long computeStepDelayMicros(float v_cm_s) {
  if (v_cm_s < V_MIN_CM_S) v_cm_s = V_MIN_CM_S;
  if (v_cm_s > V_MAX_CM_S) v_cm_s = V_MAX_CM_S;

  float stepsPerSecond = v_cm_s * STEPS_PER_CM;     // [steps/s]
  if (stepsPerSecond < 1.0f) stepsPerSecond = 1.0f;

  float periodPerStep  = 1.0e6f / stepsPerSecond;   // µs pro STEP-Zyklus
  unsigned long delayMicros = (unsigned long)(periodPerStep / 2.0f);
  if (delayMicros < 1UL) delayMicros = 1UL;
  return delayMicros;
}

// aus aktueller Fahrgeschwindigkeit den STEP_DELAY_MICROS berechnen
void updateStepDelayFromSpeed() {
  STEP_DELAY_MICROS = computeStepDelayMicros(current_speed_cm_s);
}

// --------- Endstop entprellt lesen ----------
bool readEndstopDebounced(uint8_t pin) {
  static unsigned long tLow1 = 0;
  static unsigned long tLow2 = 0;

  unsigned long now = millis();
  bool rawLow = (digitalRead(pin) == LOW);

  unsigned long &tLow = (pin == ENDSTOP_1_PIN) ? tLow1 : tLow2;

  if (rawLow) {
    if (tLow == 0) tLow = now;
    return (now - tLow) >= ENDSTOP_DEBOUNCE_MS;
  } else {
    tLow = 0;
    return false;
  }
}

#if DEBUG_ENDSTOPS
void debugPrintEndstops(bool end1Deb, bool end2Deb) {
  static unsigned long lastDbg = 0;
  unsigned long now = millis();
  if (now - lastDbg < DEBUG_INTERVAL_MS) return;
  lastDbg = now;

  int raw1 = digitalRead(ENDSTOP_1_PIN);
  int raw2 = digitalRead(ENDSTOP_2_PIN);

  Serial.print("DBG;raw1=");
  Serial.print(raw1);
  Serial.print(";raw2=");
  Serial.print(raw2);
  Serial.print(";deb1=");
  Serial.print(end1Deb ? 1 : 0);
  Serial.print(";deb2=");
  Serial.print(end2Deb ? 1 : 0);
  Serial.print(";state=");
  Serial.print((int)state);
  Serial.print(";pos=");
  Serial.println(currentPositionSteps);
}
#endif

// Ein Step mit frei wählbarem Delay (für Homing / Backoff)
void doStepImmediateWithDelay(bool directionUp, unsigned long delayMicros) {
  digitalWrite(DIR_PIN, directionUp ? HIGH : LOW);
  digitalWrite(STEP_PIN, HIGH);
  delayMicroseconds(delayMicros);
  digitalWrite(STEP_PIN, LOW);
  delayMicroseconds(delayMicros);

  if (directionUp) {
    currentPositionSteps++;
  } else {
    currentPositionSteps--;
  }
}

// Ein Step ohne Rampe mit normaler Geschwindigkeit
void doStepImmediate(bool directionUp) {
  doStepImmediateWithDelay(directionUp, STEP_DELAY_MICROS);
}

// Ein Step mit Rampe (Jog & Goto, normale Fahrten)
void doStepRamp(bool directionUp) {
  digitalWrite(DIR_PIN, directionUp ? HIGH : LOW);

  // Wenn gerade neu gestartet: langsam beginnen
  if (!rampActive) {
    unsigned long startDelay = STEP_DELAY_MICROS * (unsigned long)RAMP_START_FACTOR;
    if (startDelay < STEP_DELAY_MICROS) {
      startDelay = STEP_DELAY_MICROS;
    }
    currentStepDelayMicros = startDelay;
    rampActive = true;
  } else {
    // weiche Annäherung an Ziel-Delay
    currentStepDelayMicros =
      ( (currentStepDelayMicros * (RAMP_SMOOTH - 1)) + STEP_DELAY_MICROS ) / RAMP_SMOOTH;
  }

  digitalWrite(STEP_PIN, HIGH);
  delayMicroseconds(currentStepDelayMicros);
  digitalWrite(STEP_PIN, LOW);
  delayMicroseconds(currentStepDelayMicros);

  if (directionUp) {
    currentPositionSteps++;
  } else {
    currentPositionSteps--;
  }
}

void updateLeds(bool isMoving, bool faultActive) {
  unsigned long now = millis();

  // GRÜN: an, wenn NICHT in Bewegung und KEIN Fehler
  digitalWrite(LED_GREEN, (!isMoving && !faultActive) ? HIGH : LOW);

  // GELB: blinkt langsam, solange Bewegung
  if (isMoving) {
    if (now - lastYellowToggle >= YELLOW_PERIOD_MS) {
      lastYellowToggle = now;
      yellowState = !yellowState;
      digitalWrite(LED_YELLOW, yellowState ? HIGH : LOW);
    }
  } else {
    yellowState = false;
    digitalWrite(LED_YELLOW, LOW);
  }

  // ROT: blinkt schnell, wenn irgendein Fehlerzustand aktiv ist
  if (faultActive) {
    if (now - lastRedToggle >= RED_PERIOD_MS) {
      lastRedToggle = now;
      redState = !redState;
      digitalWrite(LED_RED, redState ? HIGH : LOW);
    }
  } else {
    redState = false;
    digitalWrite(LED_RED, LOW);
  }
}

const char* stateToString(State s) {
  switch (s) {
    case STATE_NORMAL:        return "STATE_NORMAL";
    case STATE_BACKOFF:       return "STATE_BACKOFF";
    case STATE_FAULT_LATCHED: return "STATE_FAULT";
    case STATE_HOMING:        return "STATE_HOMING";
    case STATE_SERVICE_HOMING:return "STATE_SERVICE";
  }
  return "UNKNOWN";
}

void sendStatus(bool isMoving, bool faultActive,
                bool end1Active, bool end2Active) {
  unsigned long now = millis();
  if (now - lastStatusMs < STATUS_INTERVAL_MS) return;
  lastStatusMs = now;

  Serial.print("STATUS;");
  Serial.print(stateToString(state));
  Serial.print(';');
  Serial.print(isMoving ? 1 : 0);
  Serial.print(';');
  Serial.print(faultActive ? 1 : 0);
  Serial.print(';');
  Serial.print(end1Active ? 1 : 0);
  Serial.print(';');
  Serial.print(end2Active ? 1 : 0);
  Serial.print(';');
  Serial.println(currentPositionSteps);
}

// -------------------- SERIELLE EINGÄNGE --------------------

void handleSerial() {
  while (Serial.available() > 0) {
    char c = Serial.read();

    if (c == 'U') {        // Jog AUF starten
      remoteJogUp = true;
      remoteJogDown = false;
    }
    else if (c == 'u') {   // Jog AUF stoppen
      remoteJogUp = false;
    }
    else if (c == 'D') {   // Jog AB starten
      remoteJogDown = true;
      remoteJogUp = false;
    }
    else if (c == 'd') {   // Jog AB stoppen
      remoteJogDown = false;
    }
    else if (c == 'S') {   // Sofort-Stopp
      remoteJogUp = remoteJogDown = false;
      gotoActive = false;
      // Homing/Service ggf. abbrechen
      if (state == STATE_HOMING || state == STATE_SERVICE_HOMING) {
        homingActive = false;
        state = STATE_NORMAL;
      }
    }
    else if (c == 'Q') {   // Fehler quittieren (GUI)
      if (state == STATE_FAULT_LATCHED) {
        bool e1 = readEndstopDebounced(ENDSTOP_1_PIN);
        bool e2 = readEndstopDebounced(ENDSTOP_2_PIN);
        if (!e1 && !e2) {
          state = STATE_NORMAL;
        }
      }
    }
    else if (c == 'H') {   // Homing (Referenzfahrt nach oben)
      if (state != STATE_FAULT_LATCHED) {
        remoteJogUp   = false;
        remoteJogDown = false;
        gotoActive    = false;

        homingActive    = true;
        homingStepsDone = 0;
        rampActive      = false;

        homed           = false;   // solange Homing läuft, noch nicht „homed“
        state = STATE_HOMING;
      }
    }

    else if (c == 'L') {   // Servicefahrt nach unten (unterer Endschalter)
      if (state != STATE_FAULT_LATCHED) {
        remoteJogUp   = false;
        remoteJogDown = false;
        gotoActive    = false;

        homingActive    = true;
        homingStepsDone = 0;
        rampActive      = false;

        // Service setzt KEINEN Nullpunkt und verändert 'homed' nicht
        state = STATE_SERVICE_HOMING;
      }
    }
    else if (c == 'V') {
      // V<cm/s>\n
      String vStr = Serial.readStringUntil('\n');
      vStr.trim();
      vStr.replace(',', '.');
      float v = vStr.toFloat();
      if (v < V_MIN_CM_S) v = V_MIN_CM_S;
      if (v > V_MAX_CM_S) v = V_MAX_CM_S;
      current_speed_cm_s = v;
      updateStepDelayFromSpeed();
    }
    else if (c == 'G') {
      // G<steps>\n – absolute Position in Steps
      String pStr = Serial.readStringUntil('\n');
      pStr.trim();
      long target = pStr.toInt();

      // Nur erlauben, wenn Achse referenziert wurde
      if (!homed) {
        Serial.println("ERR;NOT_HOMED");
        return;
      }

      gotoTargetSteps = target;
      gotoActive = true;
      remoteJogUp = remoteJogDown = false;
    }
  }
}

// -------------------- SETUP --------------------

void setup() {
  pinMode(STEP_PIN, OUTPUT);
  pinMode(DIR_PIN, OUTPUT);

  pinMode(JOG_UP_PIN, INPUT_PULLUP);
  pinMode(JOG_DOWN_PIN, INPUT_PULLUP);

  pinMode(ENDSTOP_1_PIN, INPUT_PULLUP);
  pinMode(ENDSTOP_2_PIN, INPUT_PULLUP);

  pinMode(LED_GREEN, OUTPUT);
  pinMode(LED_YELLOW, OUTPUT);
  pinMode(LED_RED, OUTPUT);
  pinMode(LED_STATUS, OUTPUT);

  digitalWrite(LED_GREEN, HIGH);   // steht / OK
  digitalWrite(LED_STATUS, HIGH);  // Arduino hat Versorgung

  updateStepDelayFromSpeed();
  homingStepDelayMicros = computeStepDelayMicros(HOMING_SPEED_CM_S);

  currentStepDelayMicros = STEP_DELAY_MICROS;
  rampActive = false;

  homed = false;

  Serial.begin(115200);

#if DEBUG_ENDSTOPS
  delay(200);
  Serial.println("DBG;ENABLED");
#endif
}

// -------------------- LOOP --------------------

void loop() {
  handleSerial();

  // Hardware-Eingänge
  bool jogUpHW   = (digitalRead(JOG_UP_PIN)   == LOW);
  bool jogDownHW = (digitalRead(JOG_DOWN_PIN) == LOW);

  bool jogUp   = jogUpHW   || remoteJogUp;
  bool jogDown = jogDownHW || remoteJogDown;

  // Endschalter (INPUT_PULLUP => LOW = ausgelöst) – entprellt
  bool end1Active = readEndstopDebounced(ENDSTOP_1_PIN);
  bool end2Active = readEndstopDebounced(ENDSTOP_2_PIN);

#if DEBUG_ENDSTOPS
  debugPrintEndstops(end1Active, end2Active);
#endif

  bool isMoving    = false;
  bool faultActive = (state == STATE_FAULT_LATCHED || (state == STATE_BACKOFF && backoffIsFault));

  // Wenn gerade keine normale Bewegung angefordert ist, Rampe zurücksetzen
  if (!(state == STATE_NORMAL && (jogUp || jogDown || gotoActive))) {
    rampActive = false;
  }

  switch (state) {

    case STATE_NORMAL: {

      // Sicherheitsnetz: Endschalter aktiv + Fahrbefehl -> Backoff
      if ((end1Active || end2Active) && (jogUp || jogDown || gotoActive)) {
        remoteJogUp = remoteJogDown = false;
        gotoActive = false;

        if (end2Active) {
          backoffDirectionUp = false; // nach unten weg von oben
        } else if (end1Active) {
          backoffDirectionUp = true;  // nach oben weg von unten
        } else {
          backoffDirectionUp = !lastDirectionUp;
        }

        backoffIsFault  = true;
        backoffRemaining = BACKOFF_STEPS;
        state = STATE_BACKOFF;
        break;
      }

      // GOTO hat Vorrang
      if (gotoActive) {
        if (currentPositionSteps == gotoTargetSteps) {
          gotoActive = false;
        } else {
          bool dirUp = (gotoTargetSteps > currentPositionSteps);
          lastDirectionUp = dirUp;

          if (dirUp && end2Active) {
            remoteJogUp = remoteJogDown = false;
            gotoActive = false;
            backoffDirectionUp = false;
            backoffIsFault     = true;
            backoffRemaining   = BACKOFF_STEPS;
            state = STATE_BACKOFF;
          } else if (!dirUp && end1Active) {
            remoteJogUp = remoteJogDown = false;
            gotoActive = false;
            backoffDirectionUp = true;
            backoffIsFault     = true;
            backoffRemaining   = BACKOFF_STEPS;
            state = STATE_BACKOFF;
          } else {
            doStepRamp(dirUp);
            isMoving = true;
          }
        }
      }
      else {
        // Jog-Fahrten
        if (jogUp && !jogDown) {
          lastDirectionUp = true;
          if (end2Active) {
            remoteJogUp = remoteJogDown = false;
            backoffDirectionUp = false;
            backoffIsFault     = true;
            backoffRemaining   = BACKOFF_STEPS;
            state = STATE_BACKOFF;
          } else {
            doStepRamp(true);
            isMoving = true;
          }
        }
        else if (jogDown && !jogUp) {
          lastDirectionUp = false;
          if (end1Active) {
            remoteJogUp = remoteJogDown = false;
            backoffDirectionUp = true;
            backoffIsFault     = true;
            backoffRemaining   = BACKOFF_STEPS;
            state = STATE_BACKOFF;
          } else {
            doStepRamp(false);
            isMoving = true;
          }
        }
      }

      faultActive = false;
      break;
    }

    case STATE_HOMING: {

      faultActive = false;

      if (!homingActive) {
        state = STATE_NORMAL;
        break;
      }

      // Oberer Endschalter (ENDSTOP_2_PIN) ist Referenz
      if (!end2Active) {
        // HOMING-FAHRT mit fester Geschwindigkeit
        doStepImmediateWithDelay(true, homingStepDelayMicros);
        isMoving = true;

        homingStepsDone++;
        if (homingStepsDone > HOMING_MAX_STEPS) {
          homingActive   = false;
          backoffIsFault = true;
          state          = STATE_FAULT_LATCHED;
        }
      } else {
        // Oberer Endschalter erreicht -> Nullpunkt
        currentPositionSteps = 0;

        homingActive = false;

        // kleines Stück nach unten vom Schalter wegfahren
        backoffDirectionUp = false;       // nach unten weg von oben
        backoffRemaining   = BACKOFF_STEPS;
        backoffIsFault     = false;       // KEIN Fehler, nur Homing-Backoff
        backoffSetHomed    = true;        // am Ende Backoff -> homed=true
        state              = STATE_BACKOFF;

        remoteJogUp  = false;
        remoteJogDown = false;
        gotoActive   = false;
      }

      break;
    }

    case STATE_SERVICE_HOMING: {

      faultActive = false;

      if (!homingActive) {
        state = STATE_NORMAL;
        break;
      }

      // Unterer Endschalter (ENDSTOP_1_PIN) anfahren – Servicefahrt
      if (!end1Active) {
        // SERVICE-FAHRT mit fester Homing-Geschwindigkeit nach unten
        doStepImmediateWithDelay(false, homingStepDelayMicros);
        isMoving = true;

        homingStepsDone++;
        if (homingStepsDone > HOMING_MAX_STEPS) {
          homingActive   = false;
          backoffIsFault = true;
          state          = STATE_FAULT_LATCHED;
        }
      } else {
        // Unterer Endschalter erreicht -> KEIN Nullpunkt, nur wegfahren
        homingActive = false;

        // kleines Stück nach oben vom Schalter wegfahren
        backoffDirectionUp = true;        // nach oben weg von unten
        backoffRemaining   = BACKOFF_STEPS;
        backoffIsFault     = false;       // KEIN Fehler, nur Service-Backoff
        backoffSetHomed    = false;       // Service darf homed nicht setzen
        state              = STATE_BACKOFF;

        remoteJogUp  = false;
        remoteJogDown = false;
        gotoActive   = false;
      }

      break;
    }

    case STATE_BACKOFF: {

      faultActive = backoffIsFault;

      if (backoffRemaining > 0) {
        if (backoffIsFault) {
          // Fehler-Backoff: normale Geschwindigkeit
          doStepImmediate(backoffDirectionUp);
        } else {
          // Homing-Backoff: Homing-Geschwindigkeit
          doStepImmediateWithDelay(backoffDirectionUp, homingStepDelayMicros);
        }
        backoffRemaining--;
        isMoving = true;
      } else {
        if (backoffIsFault) {
          state = STATE_FAULT_LATCHED;
        } else {
          if (backoffSetHomed) {
            homed = true;       // Homing erfolgreich abgeschlossen
          }
          state = STATE_NORMAL;
        }
      }
      break;
    }

    case STATE_FAULT_LATCHED: {

      faultActive = true;

      bool jogAny = jogUp || jogDown;

      // Hardware-Quittierung: Wippschalter 5 s halten,
      // nur wenn Endschalter frei
      if (jogAny && !end1Active && !end2Active) {
        if (faultAckStartTime == 0) {
          faultAckStartTime = millis();
        } else {
          if (millis() - faultAckStartTime >= FAULT_ACK_TIME_MS) {
            state = STATE_NORMAL;
            faultAckStartTime = 0;
          }
        }
      } else {
        faultAckStartTime = 0;
      }
      break;
    }
  }

  updateLeds(isMoving, faultActive);
  sendStatus(isMoving, faultActive, end1Active, end2Active);
}
